;; ##################################################################################
;; # ============================================================================== #
;; # ABC - editors.ss                                                               #
;; # http://abc.lozi.org                                                            #
;; # Copyright (C) Lozi Jean-Pierre, 2004 - mailto:jean-pierre@lozi.org             #
;; # ============================================================================== #
;; #                                                                                #
;; # This program is free software; you can redistribute it and/or                  #
;; # modify it under the terms of the GNU General Public License                    #
;; # as published by the Free Software Foundation; either version 2                 #
;; # of the License, or (at your option) any later version.                         #
;; #                                                                                #
;; # This program is distributed in the hope that it will be useful,                #
;; # but WITHOUT ANY WARRANTY; without even the implied warranty of                 #
;; # MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                  #
;; # GNU General Public License for more details.                                   #
;; #                                                                                #
;; # You should have received a copy of the GNU General Public License              #
;; # along with this program; if not, write to the Free Software                    #
;; # Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.    #
;; #                                                                                #
;; ##################################################################################

;; This module contains some derived editor/editor canvas declarations.
(module editors mzscheme
  
  ;; Requirements
  (require (lib "mred.ss" "mred")
           "line-numbers-widget.ss"
           (lib "string.ss" "srfi" "13")
           (lib "class.ss"))
  
  ;; This style is the line numbers' text style.
  (define normal-style (make-object style-delta% 'change-family 'modern))
  (send normal-style set-size-in-pixels-on #t)
  (send normal-style set-delta 'change-size 12)
  (send normal-style set-delta-foreground (make-object color% 0 0 0))
  (send normal-style set-weight-on 'normal)
  (send normal-style set-style-on 'normal)
  
  ;; This function returns a list containing the number of each displayed line.
  (define (get-displayed-lines-list editor)
    ;; We box the start and end variable so that they can be updated by get-visible-line-range...
    (let ((start (box 0)) (end (box 0)) (list '()))
      ;; ...we call get-visible-line-range...
      (send editor get-visible-line-range start end #f)
      ;; ...for each line...
      (do ((i (unbox end) (sub1 i)))
        ;; ...if no line remains, we return the list...
        ((< i (unbox start)) list)
        ;; ...otherwise we update the list with the current index.
        (set! list (cons (number->string i) list)))))
  
  ;; This function updates the lines number from the "from" editor to the "to" widget.
  (define (update-line-numbers from to)
    ;; We first get the lines list...
    (let ((lines (get-displayed-lines-list from)))
      ;; ...if the current list and the new list are not equal...
      (if (not (equal? (send to get-line-numbers-list) lines))
          ;; ...we update the "to" widget...
          (begin (send to set-line-numbers-list (get-displayed-lines-list from))
                 ;; ...and call its on-paint function.
                 (send to on-paint)))))
  
  ;; Provided objects
  (provide editor-text% editor-editor-canvas%)
  
  ;; We define editor-editor-canvas%...
  (define editor-editor-canvas%
    ;; ...which derives from an editor-canvas% class.
    (class editor-canvas%
      ;; Init fields
      (init-field (line-numbers-widget #f) (editor-text #f))
      ;; Overriden methods
      (override on-paint on-size)
      ;; Renamings
      (rename (super-on-paint on-paint))
      ;; We update the on-paint and on-size event handlers to update the line numbers.
      (define (on-paint) (update-line-numbers editor-text line-numbers-widget) (super-on-paint))
      (define (on-size width height) (update-line-numbers editor-text line-numbers-widget))
      ;; Parent class initialization
      (super-new)))
  
  ;; We define editor-text%...
  (define editor-text%
    ;; ...which derives from a text% class.
    (class text%
      
      ;; Global variables
      (define no-change-since-last-save #t)
      
      ;; Overriden methods.
      (override after-insert on-default-char)
      
      ;; Renamed methods.
      (rename (super-on-default-char on-default-char))
      
      ;; Public methods.
      (public indent-all indent-selection)
      
      ;; Init fields.
      (init-field (line-numbers-widget #f) (frame #f))
      
      ;; This style is the normal text style.
      (define normal-style (make-object style-delta% 'change-family 'modern))
      (send normal-style set-size-in-pixels-on #t)
      (send normal-style set-delta 'change-size 12)
      (send normal-style set-delta-foreground (make-object color% 0 0 0))
      (send normal-style set-weight-on 'normal)
      (send normal-style set-style-on 'normal)
      
      ;; The text style used by keywords.
      (define keywords-style (make-object style-delta% 'change-family 'modern))
      (send keywords-style set-delta-foreground (make-object color% 0 0 255))
      (send keywords-style set-style-on 'normal)
      
      ;; The text style used by operators.
      (define operators-style (make-object style-delta% 'change-family 'modern))
      (send operators-style set-delta-foreground (make-object color% 255 0 255))
      (send operators-style set-style-on 'normal)
      
      ;; The text style used by comments.
      (define comments-style (make-object style-delta% 'change-family 'modern))
      (send comments-style set-delta-foreground (make-object color% 0 128 0))
      (send comments-style set-style-on 'italic)
      
      ;; The text style used by strings.
      (define strings-style (make-object style-delta% 'change-family 'modern))
      (send strings-style set-delta-foreground (make-object color% 255 0 0))
      (send strings-style set-style-on 'normal)
      
      ;; Keywords.
      (define keywords '("LET" "LABEL" "FOR" "TO" "STEP" "NEXT" "DIM" "IF" "THEN" "ELSE" "GOTO" "LOCAL" "RETURN" "INPUT" "TRUE" "FALSE"
                               "DEF" "PRINT" "END" "GOSUB" "SUB" "LOCALS" "ENDSUB" "AND" "OR" "INT" "SQR" "ABS" "SIN" "COS"
                               "ASIN" "ACOS" "EXP" "LOG" "RND" "MOD" "GRAPHICS" "RESIZE" "MOVE" "PLOT" "LINETO" "COLOR"
                               "WAIT" "RECTANGLE" "ELLIPSE" "FILLEDRECTANGLE" "FILLEDELLIPSE" "LEFT$" "RIGHT$" "MID$" "STR$" "CHR$" "ASC" "LEN"))
      
      ;; Operators.
      (define operators '("=" "+" "-" "*" "/" "^" "<" ">" "<=" ">=" "<>"))
      
      ;; This function finds the depth for the line no. line.
      (define (get-line-depth line)
        ;; First we initialize the depth.
        (let ((depth 0))
          ;; For i from 1 to the current line...
          (do ((i 1 (add1 i)))
            ;; ...if we reached the given line, we just return the depth.
            ((> i line) (max depth 0))
            ;; Otherwise, we find the location of the beginning of the line...
            (let ((loc (inexact->exact (send this line-start-position i))))
              ;; ...we store the current line and the previous line...
              (let ((prev-str (string-trim (send this get-text (send this line-start-position (sub1 i)) (send this line-end-position (sub1 i)))))
                    (str (string-trim (send this get-text (send this line-start-position i) (send this line-end-position i)))))
                ;; ...if the previous line is either a SUB or a FOR instruction...
                (if (or (and (>= (string-length prev-str) 3) (string=? (substring prev-str 0 3) "SUB"))
                        (and (>= (string-length prev-str) 3) (string=? (substring prev-str 0 3) "FOR")))
                    ;; ...we increment the depth.
                    (set! depth (add1 depth)))
                ;; If the previous line is either an ENSUB or NEXT instruction...
                (if (or (and (>= (string-length str) 6) (string=? (substring str 0 6) "ENDSUB"))
                        (and (>= (string-length str) 4)  (string=? (substring str 0 4) "NEXT")))
                    ;; ...we decrement the depth.
                    (set! depth (sub1 depth))))))))
      
      ;; This function sets the given alignment to the given line.
      (define (set-line-alignment i alignment)
        ;; We store the start and end positions of the line, and the string.
        (let ((start (send this line-start-position i))
              (end (send this line-end-position i))
              (str (string-trim (send this get-text (send this line-start-position i) (send this line-end-position i)))))
          ;; Then we insert as much spaces as needed at the beginning of the line.
          (send this insert (string-append (string-pad " " (* alignment 5)) str) start end)))
      
      ;; This function indents between the given lines.
      (define (indent-lines start end)
        ;; For i starting at the first line...
        (do ((i start (add1 i)))
          ;; ...if we reached the end whe have nothing more to do.
          ((> i end) (void))
          ;; Otherwise, we set the right line alignment.
          (set-line-alignment i (get-line-depth i))))
      
      ;; This function indents the current selection.
      (define (indent-selection)
        ;; We store the beginning and ending line of the selection.
        (let* ((start (send this get-start-position))
               (end (send this get-end-position))
               (start-line (send this position-line (send this get-start-position)))
               (end-line (send this position-line (send this get-end-position))))
          ;; Then we indent the lines.
          (indent-lines start-line end-line)
          (send this begin-edit-sequence)
          (send this set-position start start #f #f 'default)
          (colorize-syntax)
          (send this set-position start end #f #f 'default)
          (send this end-edit-sequence)))
      
      ;; This function indents all the text from the widget 
      (define (indent-all)
        ;; We store the beginning and ending line of the text.
        (let ((start-line (send this position-line 0))
              (end-line (send this position-line (send this last-position))))
          ;; Then we indent the lines.
          (indent-lines start-line end-line)))
      
      ;; The overriden on-default-char method.
      (define (on-default-char event)
        ;; There have been changes since last change...
        (send frame set-no-change-since-last-save #f)
        ;; If the char is a tab...
        (if (eq? (send event get-key-code) #\tab)
            ;; ...we indent the code correctly.
              (indent-selection)
            ;; Otherwise, we call the parent method.
            (super-on-default-char event)))
      
      ;; We override the after-insert method.
      (define (after-insert start len)
        ;; If only one character has been inserted...
        (if (and (= len 1))
            ;; ...and this character is a new line...
            (if (eq? (send this get-character start)  #\newline)
                ;; ...then we first get the current line's position...
                (let* ((position (send this position-line start)))
                  ;; ...and we update the current and next line's position.
                  (set-line-alignment (add1 position) (get-line-depth (add1 position)))
                  (set-line-alignment position (get-line-depth position)))))
        
        ;; In any case, we colorize the syntax
        (colorize-syntax))
    
    ;; This function colorizes the keywords.
    (define (colorize-keyword keyword style)
      (for-each (lambda (s) (send this change-style style s (+ s (string-length keyword))))
                (send this find-string-all keyword 'forward 0 (send this last-position) #t #f)))
    
    (define (colorize-syntax)
      ;; This function colorizes the whole file's syntax.
      ;; We first begin the edit sequence.
      (send this begin-edit-sequence)
      
      ;; We set the text to its normal style.
      (send this change-style normal-style 0 (send this last-position))
      
      ;; We colorize the keywords..
      (for-each (lambda (keyword) (colorize-keyword keyword keywords-style)) keywords)
      
      ;; ...the operators...
      (for-each (lambda (keyword) (colorize-keyword keyword operators-style)) operators)
      
      ;; ...the comments...
      (let* ((comment-indexes (send this find-string-all "REM" 'forward 0 'eof #t #f))
             ;; ...using the end of line indexes following the REM keyword.
             (all-indexes (map (lambda (i) (cons i (send this line-end-position (send this position-line i)))) comment-indexes)))
        (for-each (lambda (pair) (send this change-style comments-style (car pair) (cdr pair))) all-indexes))
      
      ;; ...the quoted strings...
      (let* ((quote-indexes (send this find-string-all "\"" 'forward 0 'eof #t #f)))
        ;; ...this function groups the element by pairs in a list...
        (letrec ((group-by-pairs 
                  ;; ...taking a list as a parameter...
                  (lambda (L)
                    ;; ...if the list is empty, we have nothing to do...
                    (cond ((null? L) L)
                          ;; ...if the list length is 1, we fill se second part of the pair
                          ;; with a fake symbol...
                          ((= (length L) 1) (list (cons (car L) 'end)))
                          ;; ...otherwise, we create a pair with the two first elements.
                          (else (cons (cons (car L) (cadr L)) (group-by-pairs (cddr L))))))))
          ;; For each pair, we change the style from the first element of the pair (containing
          ;; the start of the string), and the second element of the pair (containing the end of the
          ;; string).
          (for-each (lambda (pair) (send this change-style strings-style (car pair)
                                         ;; If the second part of the pair is the fake symbol...
                                         (if (eq? (cdr pair) 'end)
                                             ;; ...we colorize the text from the start position to the
                                             ;; end of the file.
                                             (send this last-position)
                                             ;; Otherwise, we colorize the text from the start position to the
                                             ;; end position.
                                             (add1 (cdr pair)))))
                    ;; And we apply the for-each on the quote-indexes, grouped by paris.
                    (group-by-pairs quote-indexes))))
      
      ;; ...update the line numbers...
      (update-line-numbers this line-numbers-widget)
      ;; ...and set the normal style back (because the new text always use the normal style).
      (send this change-style normal-style 'start 'end #f)
      ;; Finally, we end the edit sequence.
      (send this end-edit-sequence))
    (super-new))))