;; ##################################################################################
;; # ============================================================================== #
;; # ABC - lexer-parser.ss                                                          #
;; # http://abc.lozi.org                                                            #
;; # Copyright (C) Lozi Jean-Pierre, 2004 - mailto:jean-pierre@lozi.org             #
;; # ============================================================================== #
;; #                                                                                #
;; # This program is free software; you can redistribute it and/or                  #
;; # modify it under the terms of the GNU General Public License                    #
;; # as published by the Free Software Foundation; either version 2                 #
;; # of the License, or (at your option) any later version.                         #
;; #                                                                                #
;; # This program is distributed in the hope that it will be useful,                #
;; # but WITHOUT ANY WARRANTY; without even the implied warranty of                 #
;; # MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                  #
;; # GNU General Public License for more details.                                   #
;; #                                                                                #
;; # You should have received a copy of the GNU General Public License              #
;; # along with this program; if not, write to the Free Software                    #
;; # Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.    #
;; #                                                                                #
;; ##################################################################################

;; This module contains all the graphic functions.
(module graphics mzscheme
  (require (lib "mred.ss" "mred")
           (lib "class.ss"))
  
  ;; Public functions / variables
  (provide init resize plot rectangle ellipse filled-rectangle filled-ellipse move line-to set-color-by-rgb-values set-color-by-name wait)
  
  ;; Structures
  (define-struct point (x y))
  
  ;; Global variables
  (define global-frame '?)
  (define global-canvas '?)
  (define global-bitmap '?)
  (define global-bitmap-dc '?)
  (define global-coordinates (make-point 0 0))
  (define global-color '?)
  (define global-standard-colors 
    (list (cons "black" (make-object color% 0 0 0))
          (cons "white" (make-object color% 255 255 255))
          (cons "red" (make-object color% 255 0 0))
          (cons "green" (make-object color% 0 255 0))
          (cons "blue" (make-object color% 0 0 255))
          (cons "yellow" (make-object color% 255 255 0))
          (cons "gray" (make-object color% 128 128 128))))
          
  
  ;; This function initializes the graphic window.
  (define (init width height)
    ;; We create a frame...
    (let* ((frame (new frame% (parent #f) (label "GRAPHICS") (width width) (height height)
                       (stretchable-width #f)
                       (stretchable-height #f)))
           ;; ...a canvas...
           (canvas (new canvas% (parent frame) (style '(border)) (paint-callback on-paint)
                        (label "Canvas") (enabled #t) (min-width width) (min-height height) (stretchable-width #f)
                        (stretchable-height #f)))
           ;; ...a bitmap...
           (bitmap (make-object bitmap% width height #f))
           ;; ...a bitmap-dc...
           (bitmap-dc (new bitmap-dc% (bitmap bitmap))))
      
      ;; ...store them into global-graphics...
      (set! global-frame frame)
      (set! global-canvas canvas)
      (set! global-bitmap bitmap)
      (set! global-bitmap-dc bitmap-dc)
      (set! global-color (make-object color% 255 255 255))
      
      ;; ...initialize the dc...
      (send bitmap-dc set-background (make-object color% 0 0 0))
      (send bitmap-dc clear)
      (send bitmap-dc set-pen (make-object pen% (make-object color% 255 255 255) 1 'solid))
      (send canvas on-paint)
      
      ;; ...and show the frame.
      (send frame show #t)))
    
  ;; This function resizes the graphic window.
  (define (resize width height)
    ;; We create a new bitmap...
    (let* ((bitmap (new bitmap% (width width) (height height) (monochrome? #f)))
           ;; ...and a new bitmap-dc.
           (bitmap-dc (new bitmap-dc%  (bitmap bitmap))))
      ;; We resize the frame...
      (send global-frame resize width height)
      ;; ...and the canvas.
      (send global-canvas min-width width)
      (send global-canvas min-height height)
      ;; Then we update the global-graphics list with the new bitmap...
      (set! global-bitmap bitmap)
      ;; ...and bitmap-dc.
      (set! global-bitmap-dc bitmap-dc)))
  
  ;; This function draws a point at the current coordinates.
  (define (plot)
      ;; We draw the point into the bitmap...
      (send global-bitmap-dc draw-point
            (point-x global-coordinates)
            (point-y global-coordinates))
      ;; ...and redraw the graphic area.
      (send global-canvas on-paint))
  
  ;; This function draws a rectangle with the given width and height
  ;; at the current coordinates, and moves the current coordinates
  ;; to the bottom right cursor.
  (define (rectangle width height)
    ;; First we get the current coordinates...
    (let ((x (point-x global-coordinates))
          (y (point-y global-coordinates)))
      ;; ...we set the right brush...
      (send global-bitmap-dc set-brush (make-object brush% global-color 'transparent))
      ;; ...then we draw the rectangle into the bitmap...
      (send global-bitmap-dc draw-rectangle x y width height)
      ;; ...and redraw the graphic area.
      (send global-canvas on-paint)))
  
  ;; This function draws a ellipse with the given width and height
  ;; at the current coordinates, and moves the current coordinates
  ;; to the bottom right cursor.
  (define (ellipse width height)
    ;; First we get the current coordinates...
    (let ((x (point-x global-coordinates))
          (y (point-y global-coordinates)))
      ;; ...we set the right brush...
      (send global-bitmap-dc set-brush (make-object brush% global-color 'transparent))
      ;; ...then we draw the ellipse into the bitmap...
      (send global-bitmap-dc draw-ellipse x y width height)
      ;; ...and redraw the graphic area.
      (send global-canvas on-paint)))
  
  ;; This function draws a rectangle with the given width and height
  ;; at the current coordinates, and moves the current coordinates
  ;; to the bottom right cursor.
  (define (filled-rectangle width height)
    ;; First we get the current coordinates...
    (let ((x (point-x global-coordinates))
          (y (point-y global-coordinates)))
      ;; ...we set the right brush...
      (send global-bitmap-dc set-brush (make-object brush% global-color 'opaque))
      ;; ...then we draw the rectangle into the bitmap...
      (send global-bitmap-dc draw-rectangle x y width height)
      ;; ...and redraw the graphic area.
      (send global-canvas on-paint)))
  
  ;; This function draws a ellipse with the given width and height
  ;; at the current coordinates, and moves the current coordinates
  ;; to the bottom right cursor.
  (define (filled-ellipse width height)
    ;; First we get the current coordinates...
    (let ((x (point-x global-coordinates))
          (y (point-y global-coordinates)))
      ;; ...we set the right brush...
      (send global-bitmap-dc set-brush (make-object brush% global-color 'opaque))
      ;; ...then we draw the ellipse into the bitmap...
      (send global-bitmap-dc draw-ellipse x y width height)
      ;; ...and redraw the graphic area.
      (send global-canvas on-paint)))
  
  ;; This function moves the point to the (x,y) coordinates...
  (define (move x y)
    ;; ...we first update the first coordinate...
    (set-point-x! global-coordinates x)
    ;; ...then the second one.
    (set-point-y! global-coordinates y))
  
  ;; This function draws a line from the current coordinates to the point
  ;; (x,y) and sets the new current coordinates to (x,y).
  (define (line-to x y)
    ;; We first get the current coordinates...
    (let ((current-x (point-x global-coordinates))
          (current-y (point-y global-coordinates)))
      ;; ...draw a line...
      (send global-bitmap-dc draw-line current-x current-y x y)
      ;; ...update the current coordinates...
      (set-point-x! global-coordinates x)
      (set-point-y! global-coordinates y)
      ;; ...and redraw the graphic area.
      (send global-canvas on-paint)))
  
  ;; This function sets the current drawing color, given three RGB values.
  (define (set-color-by-rgb-values red green blue)
    ;; We first create the color...
    (let ((color (make-object color% (inexact->exact red) (inexact->exact green) (inexact->exact blue))))
      ;; ...then set the drawing color...
      (send global-bitmap-dc set-pen 
            ;; ...using a new pen...
            (make-object pen% color 1 'solid))
      ;; ...and update the global-color variable.
      (set! global-color color)))

  ;; This function sets the current drawing color, taking the color name as only parameter.
  (define (set-color-by-name name)
    ;; We first get the color...
    (let ((color (cdr (assq name global-standard-colors))))
      ;; ..and set it...
      (send global-bitmap-dc set-pen 
            ;; ...using a new pen.
            (make-object pen% color 1 'solid))))
  
  ;; This function pauses the program for the given numbers of seconds.
  (define (wait seconds) (sleep/yield seconds))
  
  ;; This function is called wen the graphic window needs to be redrawn.
  (define (on-paint canvas control-event)
    ;; We first get the dc...
    (let ((dc (send canvas get-dc))
          ;; ...the bitmap...
          (bitmap global-bitmap))
      ;; ...and draw it into the dc.
      (send dc draw-bitmap bitmap 0 0))))