;; ##################################################################################
;; # ============================================================================== #
;; # ABC - gui.ss                                                                   #
;; # http://abc.lozi.org                                                            #
;; # Copyright (C) Lozi Jean-Pierre, 2004 - mailto:jean-pierre@lozi.org             #
;; # ============================================================================== #
;; #                                                                                #
;; # This program is free software; you can redistribute it and/or                  #
;; # modify it under the terms of the GNU General Public License                    #
;; # as published by the Free Software Foundation; either version 2                 #
;; # of the License, or (at your option) any later version.                         #
;; #                                                                                #
;; # This program is distributed in the hope that it will be useful,                #
;; # but WITHOUT ANY WARRANTY; without even the implied warranty of                 #
;; # MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                  #
;; # GNU General Public License for more details.                                   #
;; #                                                                                #
;; # You should have received a copy of the GNU General Public License              #
;; # along with this program; if not, write to the Free Software                    #
;; # Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.    #
;; #                                                                                #
;; ##################################################################################

;; The Graphical User Interface module.
(module gui mzscheme
  ;; Requirements
  (require "lexer-parser.ss"
           "interpreter.ss"
           "editors.ss"
           "line-numbers-widget.ss"
           "about-dialog.ss"
           (lib "mred.ss" "mred")
           (lib "sendurl.ss" "net")
           (lib "string.ss" "srfi" "13")
           (lib "class.ss"))
  
  ;; Public variables
  (provide abc-frame%)
  
  ;; Global functions
  (define frame-list '())
  
  ;; Each frame is an object.
  (define abc-frame%
    ;; We derive the object% class.
    (class object%
      
      ;; This function runs the interpreter on a given program.
      (define run #f)
      (define line-depths #f)
      (define current-file-path #f)
      (define no-change-since-last-save #t)
      
      ;; Init fields
      (init-field (interpreter #f))
      
      ;; Public methods
      (public gui-read gui-write set-interpreter set-no-change-since-last-save)
      
      ;; This style is the normal text style.
      (define normal-style (make-object style-delta% 'change-family 'modern))
      (send normal-style set-size-in-pixels-on #t)
      (send normal-style set-delta 'change-size 10)
      (send normal-style set-delta-foreground (make-object color% 0 0 0))
      (send normal-style set-weight-on 'normal)
      (send normal-style set-style-on 'normal)
      
      ;; This style is the error text style.
      (define error-style (make-object style-delta% 'change-family 'modern))
      (send error-style set-size-in-pixels-on #t)
      (send error-style set-delta 'change-size 10)
      (send error-style set-delta-foreground (make-object color% 255 0 0))
      (send error-style set-weight-on 'bold)
      (send error-style set-style-on 'normal)
      
      ;; This style is the code text style.
      (define code-style (make-object style-delta% 'change-family 'modern))
      (send code-style set-size-in-pixels-on #t)
      (send code-style set-delta 'change-size 10)
      (send code-style set-delta-foreground (make-object color% 0 0 255))
      (send code-style set-weight-on 'bold)
      (send code-style set-style-on 'normal)
      
      ;; This style is the info text style.
      (define info-style (make-object style-delta% 'change-family 'modern))
      (send info-style set-size-in-pixels-on #t)
      (send info-style set-delta 'change-size 10)
      (send info-style set-delta-foreground (make-object color% 0 128 0))
      (send info-style set-weight-on 'bold)
      (send info-style set-style-on 'normal)
      
      ;; This function is called by the interpreter when it needs to open a prompt.
      (define (gui-read message)
        ;; We prompt the user...
        (let ((result (get-text-from-user "BASIC APPLICATION" message frame)))
          ;; ...and return the result.
          (if result result "")))
      
      ;; This function is called by the interpreter when it needs to write to the toplevel.
      (define (gui-write str type)
        ;; We set the right text style depending on the type of text.
        (case type
          ((info) (send toplevel-text change-style info-style))
          ((normal) (send toplevel-text change-style normal-style))
          ((error) (send toplevel-text change-style error-style))
          ((code) (send toplevel-text change-style code-style)))
        ;; We write the string to the toplevel.
        (send toplevel-text insert str)
        ;; Then reset the style.
        (send toplevel-text change-style normal-style))
      
      ;; Modifiers
      ;; This function sets the interpreter variable.
      (define (set-interpreter obj) (set! interpreter obj))
      ;; This function sets the no-change-since-last-save variable.
      (define (set-no-change-since-last-save value) (set! no-change-since-last-save value))
      
      ;; We derive the frame% class to handle the on-exit and on-close event handlers.
      (define frame-with-close-exit-handlers%
        ;; We derive the frame% class.
        (class frame%       
         ;; Init-fields
         (init-field (abc-frame #f))
         
         ;; Overriden methods
         (override on-close on-exit)
         
         ;; Overriden on-close method.
         (define (on-close)
           ;; We first check whether there has been any change since the last save.
           (let ((result (if (not no-change-since-last-save)
                             (message-box/custom "Warning" 
                                                 "Would you like to save changes before quitting?"
                                                 "Yes" "No" #f this '(default=1 number-order caution) 3) 2)))
             ;; If the users answered "Yes" to the last prompt, we save the current file.
             (if (= 1 result) (send abc-frame save))))
         
         ;; Overriden on-exit method.
         (define (on-exit)
           ;; We just call on-close.
           (on-close))
          
          ;; Parent class initialization.
          (super-new)))
      
      ;; The frame.
      (define frame (new frame-with-close-exit-handlers% (label "ABC is a Basic Compiler") (abc-frame this) (width 640) (height 480)))
      
      (set! frame-list (cons frame frame-list))
      
      ;; Menu bar definition...
      (define menu-bar (new menu-bar% (parent frame)))
      
      ;; File menu definition...
      (define file-menu (new menu% (label "File") (parent menu-bar) (help-string "File menu")))
      ;; Edit menu definition...
      (define edit-menu (new menu% (label "Edit") (parent menu-bar) (help-string "Edit menu")))
      ;; Basic menu definition...
      (define basic-menu (new menu% (label "Basic") (parent menu-bar) (help-string "Basic menu")))
      ;; Help menu definition...
      (define help-menu (new menu% (label "Help") (parent menu-bar) (help-string "Help menu")))
      
      ;; ===========================================================================================================
      ;; File Menu
      ;; ===========================================================================================================
      ;; "New Frame" item definition...
      (new menu-item% (label "New &Frame") (parent file-menu) (callback (lambda (item event) (new-frame))))
      ;; Separator.
      (new separator-menu-item% (parent file-menu))
      ;; "New" item definition...
      (new menu-item% (label "&New") (shortcut #\N) (parent file-menu) (callback (lambda (item event) (new-file))))
      ;; "Open..." item definition...
      (new menu-item% (label "&Open...") (shortcut #\O) (parent file-menu) (callback (lambda (item event) (open))))
      ;; Separator.
      (new separator-menu-item% (parent file-menu))
      ;; "Examples" submenu definition...
      (define examples-sub-menu (new menu% (label "&Examples") (parent file-menu)))
      ;; ===========================================================================================================
      ;; Examples submenu
      ;; ===========================================================================================================
      ;; For each file , we create a new menu-item...
      (for-each (lambda (file-name) (new menu-item% (label file-name) (parent examples-sub-menu)
                                         ;; ...its callback open the right file...
                                         (callback (lambda (item event)
                                                     ;; ...from the examples folder.
                                                     (open-file (build-path "examples" (send item get-label)))))))
                ;; We look for the examples in the examples folder.
                (directory-list "examples"))
      
      ;; Separator.
      (new separator-menu-item% (parent file-menu))
      ;; "Save" item definition...
      (new menu-item% (label "&Save") (shortcut #\S) (parent file-menu) (callback (lambda (item event) (save))))
      ;; "Save as..." item definition...
      (new menu-item% (label "Save &as...") (parent file-menu) (callback (lambda (item event) (save-as))))
      ;; Separator.
      (new separator-menu-item% (parent file-menu))
      ;; "Close" item definition...
      (new menu-item% (label "Close") (shortcut #\W) (parent file-menu) (callback (lambda (item event) (send frame on-close))))
      ;; "Quit" item definition...
      (new menu-item% (label "Quit") (shortcut #\Q) (parent file-menu)
           ;; The quit item sends on-close to each window before quitting.
           (callback (lambda (item event) (for-each (lambda (frame) (send frame on-close) (send frame show #f)) frame-list) (exit))))
      
      ;; ===========================================================================================================
      ;; Edit Menu
      ;; ===========================================================================================================
      ;; We just append the editor-operation-menu-items to the menu.
      (append-editor-operation-menu-items edit-menu)
        
      ;; ===========================================================================================================
      ;; Basic Menu
      ;; ===========================================================================================================
      ;; "Indent all" item definition...
      (new menu-item% (label "Indent All") (shortcut #\:) (parent basic-menu) (callback (lambda (item event) (send editor-text indent-all))))
      ;; "Indent selection" item definition...
      (new menu-item% (label "Indent Selection") (shortcut #\;) (parent basic-menu) (callback (lambda (item event) (send editor-text indent-selection))))
      ;; Separator.
      (new separator-menu-item% (parent basic-menu))
      ;; "Execute" item definition...
      (new menu-item% (label "Execute") (shortcut #\E) (parent basic-menu) (callback (lambda (item event) (execute))))
      ;; "Translate" item definition...
      (new menu-item% (label "Translate") (shortcut #\T) (parent basic-menu) (callback (lambda (item event) (translate))))
      
      ;; ===========================================================================================================
      ;; Help Menu
      ;; ===========================================================================================================
      ;; "Online Documentation..." item defintion...
      (new menu-item% (label "Online Documentation...") (parent help-menu) (callback (lambda (item event) (send-url "http://www.lozi.org/abc/reference.php"))))
      ;; "About ABC..."
      (new menu-item% (label "About ABC...") (parent help-menu) (callback (lambda (item event) (about))))
      
      ;; This function is called when the user chooses New in the file menu.
      (define (new-file)
        ;; We first check whether there has been any change since the last save.
        (let ((result (if (not no-change-since-last-save)
                          (message-box/custom "Warning" 
                                              "The current file has been modified. Would you like to save changes?"
                                              "Yes" "No" "Cancel" frame '(default=1 number-order caution) 3) 2)))
          ;; If the users answered "Yes" to the last prompt, we save the current file.
          (if (= 1 result) (save))
          ;; Then we erase the text from the editor...
          (send editor-text erase)
          (set! no-change-since-last-save #t)
          (set! current-file-path #f)))
      
      ;; This function is called when the user chooses Open... in the file menu.
      (define (open)
        ;; We first check whether there has been any change since the last save.
        (let ((result (if (not no-change-since-last-save)
                          (message-box/custom "Warning" 
                                              "Do you want to save the current file before opening a new one?"
                                              "Yes" "No" "Cancel" frame '(default=1 number-order caution) 3) 2)))
          ;; If the users answered "Yes" to the last prompt, we save the current file.
          (if (= 1 result) (save))
          ;; Otherwise...
          (if (not (= 3 result))
              ;; ...we prompt the user for a file...
              (let ((result (get-file "Open..." #f #f #f "*.abc" '() '(("ABC Basic File (.abc)" "*.abc")))))
                ;; ...if the user did not cancel, we just open the file.
                (if result (open-file result))))))
      
      ;; This function opens the given file.
      (define (open-file file)
        ;; If the file exists...
        (if (file-exists? file)
            ;; ...we load the file...
            (begin (send editor-text load-file file)
                   ;; ...and update the current path.
                   (set! current-file-path file)
                   ;; There was no change since last save.
                   (set! no-change-since-last-save #t))
            ;; Otherwise, there has been an error (the file does not exist).
            (message-box "Error" "The file does not exist." frame '(ok stop))))
      
      ;; This function is called when the user chooses Save in the file menu.
      (define (save)
        ;; If a file is opened...
        (if current-file-path
            ;; ...and exists...
            (if (file-exists? current-file-path)
                ;; ...then we write to the file...
                (begin (call-with-output-file current-file-path
                         ;; ...the whole text from the editor-text object...
                         (lambda (p-out) (fprintf p-out "~a" (send editor-text get-text))) 'replace)
                       ;; ...and update the global variables.
                       (set! no-change-since-last-save #t))
                ;; Otherwise, the file does not exist anymore.
                (begin (message-box "Error" "Unable to save because the file does not exist anymore. Please check it has not been either deleted, moved or renamed." frame '(ok stop))
                       ;; Therefore we just call file-save-as.
                       (save-as)))
            ;; If the current file is new, we just call save as.
            (save-as)))
      
      ;; This function is called when the user chooses Save As... in the file menu.
      (define (save-as)
        ;; We first prompt the user...
        (let ((result (put-file "Save as..." #f #f #f "*.abc" '() '(("ABC Basic File (.abc)" "*.abc")))))
          ;; ...if the did not choose Cancel, then we write to the file...
          (if result (begin (call-with-output-file result 
                              ;; ...the whole text from the editor-text object.
                              (lambda (p-out) (fprintf p-out "~a" (send editor-text get-text)) 'replace))
                            ;; Then we update the no-change-since-last-save variable...
                            (set! no-change-since-last-save #t)
                            ;; ...and we update the project path.
                            (set! current-file-path result)))))
      
      ;; This function runs the current file.
      (define (execute)
        ;; We first clear the toplevel.
        (send toplevel-text clear)
        ;; If the file does not end with a newline...
        (if (or (< (send editor-text last-position) 1)
                (not (eq? (send editor-text get-character (sub1 (send editor-text last-position))) #\newline)))
            ;; ...we insert one...
            (send editor-text insert "\n" (send editor-text last-position) (send editor-text last-position)))
        ;; ...then we run the program.
        (send interpreter run (send editor-text get-text)))
      
      ;; This function computes the current file, and shows the result into the toplevel.
      (define (translate)
        ;; If the file does not end with a newline...
        (if (or (< (send editor-text last-position) 1)
                   (not (eq? (send editor-text get-character (sub1 (send editor-text last-position))) #\newline)))
            ;; ...we insert one...
            (send editor-text insert "\n" (send editor-text last-position) (send editor-text last-position)))
        ;; ...then we translate the program, redirecting the errors...
        (with-handlers ((exn? (lambda (exn) (gui-write (string-append (exn-message exn) "\n") 'error))))
          (let* ((vector (compute (send editor-text get-text)))
                 (len (vector-length vector)))
            ;; ...for each line...
            (do ((i 0 (add1 i)))
              ;; ...if we reached the length of the vector, we have nothing more to do...
              ((= i len) (void))
              ;; ...otherwise, we display the line.
              (gui-write (string-append "[" (number->string i) "] " (format "~a" (vector-ref vector i)) "\n") 'code)))))
      
      ;; This function gets the currently selected editor.
      (define (get-selected-editor)
        ;; We just ask each editor whether it has focus or not.
        (cond ((send editor-editor-canvas has-focus?) editor-text)
              ((send editor-editor-canvas has-focus?) toplevel-text)))
      
      ;; The main vertical panel.
      (define main-vertical-panel (new vertical-panel% (parent frame) (alignment '(left top))))
      
      ;; ===========================================================================================================
      ;; Buttons
      ;; ===========================================================================================================
      (define buttons-panel (new horizontal-panel% (parent main-vertical-panel) (border 0) (spacing 0) (stretchable-height #f)))
      
      ;; The open button.
      (define open-button (new button% (label (make-object bitmap% "images/open.png" 'png (get-panel-background))) (parent buttons-panel)
                               (callback (lambda (item event) (open)))))
      
      ;; The save button.
      (define save-button (new button% (label (make-object bitmap% "images/save.png" 'png (get-panel-background))) (parent buttons-panel)
                               (callback (lambda (item event) (save)))))
      
       ;; The indent-all button.
      (define indent-all-button (new button% (label (make-object bitmap% "images/indent-all.png" 'png (get-panel-background))) (parent buttons-panel)
                                    (callback (lambda (item event) (send editor-text indent-all)))))
      
      ;; The execute button.
      (define execute-button (new button% (label (make-object bitmap% "images/execute.png" 'png (get-panel-background))) (parent buttons-panel)
                                  (callback (lambda (item event) (execute)))))
      
      ;; The translate button.
      (define translate-button (new button% (label (make-object bitmap% "images/translate.png" 'png (get-panel-background))) (parent buttons-panel)
                                    (callback (lambda (item event) (translate)))))
      
      ;; This panel contains the editor and the line numbers widget.
      (define editor-panel (new horizontal-panel% (parent main-vertical-panel)))
      
      ;; A widget containing the line numbers.
      (define line-numbers-widget (new line-numbers-widget%
                                       (parent editor-panel)
                                       (min-width 45)
                                       (stretchable-width #f)))
      
      ;; The current file's text object.
      (define editor-text (new editor-text% (line-numbers-widget line-numbers-widget) (frame this) (line-spacing 1.0) (tab-stops null) (auto-wrap #t)))
      (send editor-text change-style normal-style 'start 'end #f)
      (send editor-text set-styles-sticky #t)
      
      ;; The current file's editor-canvas.
      (define editor-editor-canvas (new editor-editor-canvas%
                                        (parent editor-panel)
                                        (line-numbers-widget line-numbers-widget)
                                        (editor editor-text)
                                        (editor-text editor-text)
                                        (style '(no-hscroll))
                                        (min-height 200)
                                        (horizontal-inset 5)))
      
      (send line-numbers-widget set-line-spacing (send editor-text get-line-spacing))
      
      ;; Toplevel's text object.
      (define toplevel-text (new text% (line-spacing 1.0) (tab-stops null) (auto-wrap #t)))
      
      ;; Toplevel's editor-canvas.
      (define toplevel-editor-canvas (new editor-canvas%
                                          (parent main-vertical-panel)
                                          (editor toplevel-text)
                                          (horizontal-inset 5)))
      
      
      (send frame show #t)
      (gui-write "Without line numbers.\nWith SUB/ENDSUB (modified, i.e. the function return types are given by the identifier types).\n" 'info)
      (send toplevel-text change-style normal-style 'start 'end #f)
      (super-new)))
  
  ;; This function creates a new frame.
  (define (new-frame)
    ;; We create a frame...
    (let* ((start-frame (new abc-frame%))
           ;; ...an interpreter...
           (interpreter (new interpreter% (frame start-frame))))
      ;; ...and bind the interpreter with the frame...
      (send start-frame set-interpreter interpreter)))
  
  ;; We finally create a new frame.
  (new-frame))