;; ##################################################################################
;; # ============================================================================== #
;; # ABC - interpreter.ss                                                           #
;; # http://abc.lozi.org                                                            #
;; # Copyright (C) Lozi Jean-Pierre, 2004 - mailto:jean-pierre@lozi.org             #
;; # ============================================================================== #
;; #                                                                                #
;; # This program is free software; you can redistribute it and/or                  #
;; # modify it under the terms of the GNU General Public License                    #
;; # as published by the Free Software Foundation; either version 2                 #
;; # of the License, or (at your option) any later version.                         #
;; #                                                                                #
;; # This program is distributed in the hope that it will be useful,                #
;; # but WITHOUT ANY WARRANTY; without even the implied warranty of                 #
;; # MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                  #
;; # GNU General Public License for more details.                                   #
;; #                                                                                #
;; # You should have received a copy of the GNU General Public License              #
;; # along with this program; if not, write to the Free Software                    #
;; # Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.    #
;; #                                                                                #
;; ##################################################################################

;; This module contains the interpreter.
(module interpreter mzscheme 
  
  ;; Inclusions
  (require (prefix graphics- "graphics.ss")
           (prefix lexer-parser- "lexer-parser.ss")
           (lib "string.ss")
           (lib "string.ss" "srfi" "13")
           (lib "class.ss"))
  
  ;; Public functions / variables
  (provide interpreter%)
  
  ;; The intepreter is an object.
  (define interpreter% 
    ;; We derive the object% class...
    (class object%
      
      ;; Global variables
      (init-field (frame '?))
      (define line-depths '())
      (define global-env '())
      (define global-dict '?)
      (define global-labels '())
      (define global-fors '())
      (define global-subs '())
      (define global-program-vector '?)
      (define stack '?)
      (define debug #f)
      
      ;; Public functions
      (public run get-line-depths set-frame)
      
      ;; Accessors
      (define (get-line-depths) line-depths)
      
      ;; Modifiers
      (define (set-frame obj) (set! frame obj)) 
      
      ;; This function extends the given environment with the given variables and values.
      (define (extend-env Lvar Lval env)
        ;; We cannot insert an empty dictionary - let's fill it in with a fake variable instead.
        (if (null? Lvar) (cons (list (cons 'dummy 0)) env)
            ;; Otherwise, we create a new dictionary.
            (cons (map cons Lvar Lval) env)))
      
      ;; This function returns #t if the given variable is bound in the given environment.
      (define (binding var env)
        (if (null? env)
            #f
            (or (assq var (car env))
                (binding var (cdr env)))))
      
      ;; This function looks for the given variable in the given environment.
      (define (lookup var env)
        (let ((try (binding var env)))
          (if try 
              (cdr try)
              (error "[INTERPRETER] Unknown identifier" var))))
      
      ;; This function sets a binding's value.
      (define (set-binding! binding val) 
        (set-cdr! binding val))
      
      ;; This function extends the given dictionnary with the given variable and value.
      (define (extend-dictionary! symb val dict)  
        (define (iter pdict) 
          (cond ((eq? (caar pdict) symb) (set-cdr! (car pdict) val)) 
                ((null? (cdr pdict)) (set-cdr! dict (cons (cons symb val) (cdr dict))))
                (else (iter (cdr pdict)))))
        (iter dict))
      
      ;; This function initializes the global environment.
      (define (init-global-env)
        ;; The initial dictionnary defines default values for some variables.
        (let ((initial-dictionnary (list (cons 'pi 3.141592653589793)
                                         (cons 'e 2.718281828459045)
                                         (cons 'my-mark-at-this-project 0))))
          ;; We remove the global environment's content.
          (set! global-env '())
          ;; We extend the global environment with the initial dictionnary.
          (set! global-env (extend-env (map car initial-dictionnary)
                                       (map cdr initial-dictionnary)
                                       global-env))
          ;; We update the global-dict variable.
          (set! global-dict (car global-env))))
      
      
      
      ;; This function runs the given program.
      (define (run program)
        (with-handlers ((exn? (lambda (exn) (send frame gui-write (string-append (exn-message exn) "\n") 'error))))
          ;; We initialize the global environment.
          (init-global-env)
          ;; We get the program vector using the lexer/parser.
          (let ((vector (lexer-parser-compute program)))
            ;; We initialize the line-depths variable.
            (set! line-depths (make-vector (vector-length vector)))
            ;; We initialize the global labels.
            (set! global-program-vector vector)
            (let-values (((fors labels subs) (get-fors-labels-subs global-program-vector)))
              (set! global-fors fors)
              (set! global-labels labels)
              (set! global-subs subs))
            ;; We run the first instruction.
            (run-instruction-by-instruction-pointer 0 '(toplevel) #f global-env))))
      
      ;;
      ;; This function gets all the FORs, LABELs, and SUBs from the program and stores the values
      ;; associated with their related values into three different lists. 
      ;; Concerning the fors, this function allows to deal with expressions like :
      ;;
      ;; FOR i = ... TO ...
      ;; <instructions>
      ;; FOR j = ... TO ...
      ;; <instructions>
      ;; NEXT i
      ;; <instructions>
      ;; NEXT j
      ;;
      ;; And:
      ;;
      ;; FOR i = ... TO ...
      ;; SUB f( ... )
      ;; ...
      ;; FOR ...
      ;; NEXT
      ;; ...
      ;; ENDSUB
      ;; ...
      ;; NEXT
      ;;
      ;; For this last example  the NEXT refers to the FOR before the function.
      ;;
      ;; Is it useful for any purpose? I hardly believe it, but I suppose this is the right behaviour!
      ;;
      (define (get-fors-labels-subs global-program-vector)
        ;; The iterative subfunction making all the work.
        (define (iter fors labels subs functions-hierarchy i)
          ;; If we are at the last index, we just return the list.
          (if (= i (vector-length global-program-vector))
              (values (apply append (map cadr fors)) labels subs)
              ;; Otherwise...
              (let ((instruction (vector-ref global-program-vector i)))
                ;; ...we first register the line depth...
                (vector-set! line-depths i (length functions-hierarchy))
                ;; ...if we enter a subfunction...
                (cond ((eq? (car instruction) '*sub*)
                       ;; ...we store its name in the first position of functions-hierarchy, and update the subs list.
                       (let ((next-function-hierarchy (insert-function functions-hierarchy (cadr instruction))))
                         (iter fors labels (insert-sub subs i (car next-function-hierarchy)) next-function-hierarchy (+ i 1))))
                      ;; ...if we get out from a subfunction...
                      ((eq? (car instruction) '*endsub*)
                       ;; ...we remove its name from function-hierarchy and upddate the subs list.
                       (iter fors labels (update-sub subs i (car functions-hierarchy)) (cdr functions-hierarchy) (+ i 1)))
                      ;; ...if the current line is a for-to...
                      ((eq? (car instruction) '*for-to*)
                       ;; ...we insert the right variables, specifying the current function.
                       (iter (insert-for fors i (car functions-hierarchy) (cadr instruction)
                                         (caddr instruction) (cadddr instruction) 1)
                             labels subs
                             functions-hierarchy (+ i 1)))
                      ;; ...if the current line is a for-to-step...
                      ((eq? (car instruction) '*for-to-step*)
                       ;; ...we insert the right variables, specifying the current function.
                       (iter (insert-for fors i (car functions-hierarchy) (cadr instruction) (caddr instruction) (cadddr instruction) (car (cddddr instruction)))
                             labels subs functions-hierarchy (+ i 1)))
                      ;; ...if the current line is a next...
                      ((eq? (car instruction) '*next*)
                       ;; ...we update the fors list.
                       (iter (update-for fors i (car functions-hierarchy) 'any) labels subs functions-hierarchy (+ i 1)))
                      ;; ...if the current line is a next...
                      ((eq? (car instruction) '*next-identifier*)
                       ;; ...we update the fors list.
                       (iter (update-for fors i (car functions-hierarchy) (cadr instruction)) labels subs functions-hierarchy (+ i 1)))
                      ;; If we found a label, we insert it into the labels-list, and continue parsing the vector.
                      ((eq? (car instruction) '*label*)
                       (iter fors (cons (list (cadr (vector-ref global-program-vector i)) i) labels) subs functions-hierarchy (+ i 1)))
                      ;; ...otherwise, we just continue.
                      (else (iter fors labels subs functions-hierarchy (+ i 1)))))))
        
        ;; This subfunction inserts the right for variables into the L list.
        (define (insert-for fors i function variable start end step)
          ;; If we are in the end of the list, then we create a sublist for the current function.
          (cond ((null? fors) `((,function ((,i '? ,variable ,start ,end ,step)))))
                ;; Otherwise, if we found the current function, we just insert the variables list.
                ((eq? (caar fors) function) (cons (list (caar fors) (cons (list i '? variable start end step) (cadar fors))) (cdr fors)))
                ;; Otherwise, we just continue.
                (else (cons (car fors) (insert-for (cdr fors) i function variable start end step)))))
        
        ;; This function updates a fors list when we find a NEXT.
        (define (update-for fors i function variable)
          (define (sub-update L)
            ;; If we did not find the right element, we return an error.
            (cond ((null? L) (error "[INTERPRETER] Unable to interpret the FOR/NEXT structure"))
                  ;; If we found the right element, we update it.
                  ((or (eq? variable 'any) (eq? (caddar L) variable)) (cons (list (list-ref (car L) 0) i (list-ref (car L) 2) (list-ref (car L) 3) (list-ref (car L) 4) (list-ref (car L) 5)) (cdr L)))
                  ;; Otherwise, we continue searching.
                  (else (cons (car L) (sub-update (cdr L))))))
          
          ;; We first locate the right sub-list (which depends on the function...
          (cond ((null? fors) (error "[INTERPRETER] Unable to interpret the FOR/NEXT structure"))
                ;; ...and then call the sub-update function.
                ((eq? (caar fors) function) (cons (list (caar fors) (sub-update (cadar fors))) (cdr fors)))
                (else (cons (car fors) (update-for (cdr fors) i function variable)))))
        
        ;; This function allows to insert an new element into the subs list.
        (define (insert-sub subs i function) (cons (list function i '? #f) subs))
        
        ;; This function updates a fors list when we reach an ENDSUB.
        (define (update-sub subs i function)
          ;; If we did not find the right element, we return an error.
          (cond ((null? subs) (error "[INTERPRETER] Unexpected error"))
                ;; If we found the right element, we update it.
                ((eq? (caar subs) function) (cons (list (caar subs) (cadar subs) i #f) (cdr subs)))
                ;; Otherwise, we continue searching.
                (else (cons (car subs) (update-sub (cdr subs) i function)))))
        
        ;; We just call the iter function.
        (iter '((toplevel ())) '() '() '(toplevel) 0))
      
      ;; This function runs the instruction at the given instruction-pointer.
      (define (run-instruction-by-instruction-pointer instruction-pointer hierarchy additional-data env)
        ;; If we are not at the end of the vector...
        (if (< instruction-pointer (vector-length global-program-vector))
            ;; ...we get the current instruction...
            (let ((current-instruction (vector-ref global-program-vector instruction-pointer)))
              ;; ...and run it.
              (run-instruction-by-list current-instruction instruction-pointer hierarchy additional-data env))))
      
      ;; This function runs the given instruction (rawly translated by lex/yacc).
      (define (run-instruction-by-list current-instruction instruction-pointer hierarchy additional-data env)
        ;; We first get the instruction type.
        (case (car current-instruction)
          ((*eval*) (eval-expression (cadr current-instruction) hierarchy instruction-pointer env)
                    (run-instruction-by-instruction-pointer (add1 instruction-pointer) hierarchy #f env))
          ;; INPUT <string>, <numeric-identifier>
          ((*input-numeric*) (run-input-numeric (cadr current-instruction) (caddr current-instruction) env)
                             (run-instruction-by-instruction-pointer (add1 instruction-pointer) hierarchy #f env))
          
          ;; INPUT <string>, <string-identifier>
          ((*input-string*) (run-input-string (cadr current-instruction) (caddr current-instruction) env)
                            (run-instruction-by-instruction-pointer (add1 instruction-pointer) hierarchy #f env))
          
           ;; INPUT <string>, <boolean-identifier>
          ((*input-string*) (run-input-boolean (cadr current-instruction) (caddr current-instruction) env)
                            (run-instruction-by-instruction-pointer (add1 instruction-pointer) hierarchy #f env))
          
          ;; PRINT <string-list>
          ((*print*) (run-print (cdr current-instruction) hierarchy instruction-pointer #t env)
                     (run-instruction-by-instruction-pointer (add1 instruction-pointer) hierarchy #f env))
          
          ;; PRINT-SEMICOLON <string-list>
          ((*print-semicolon*) (run-print (cdr current-instruction) hierarchy instruction-pointer #f env)
                               (run-instruction-by-instruction-pointer (add1 instruction-pointer) hierarchy #f env))
          
          ;; IF <condition> THEN <instruction>
          ((*if-then*) (if (eval-expression (cadr current-instruction) hierarchy instruction-pointer env)
                           (run-instruction-by-list (caddr current-instruction) instruction-pointer hierarchy #f env)
                           (run-instruction-by-instruction-pointer (add1 instruction-pointer) hierarchy #f env)))
          
          ;; IF <condition> THEN <instruction> ELSE <instruction>
          ((*if-then-else*) (if (eval-expression (cadr current-instruction) hierarchy instruction-pointer env)
                                (run-instruction-by-list (caddr current-instruction) instruction-pointer hierarchy #f env)
                                (run-instruction-by-list (cadddr current-instruction) instruction-pointer hierarchy #f env)))
          
          ;; LET <identifier> = <numeric-expression>
          ;; LET <string-identifier> = <string-expression>
          ;; LET <boolean-identifier> = <boolean-expression>
          ((*let-numeric* *let-string* *let-boolean*) (run-let-variable (cadr current-instruction) (caddr current-instruction) hierarchy instruction-pointer env)
                                                      (run-instruction-by-instruction-pointer (add1 instruction-pointer) hierarchy #f env))
          
          ;; LET <identifier> ( <numeric-expression> ) = <numeric-expression>
          ;; LET <string-identifier> ( <numeric-expression> ) = <string-expression>
          ;; LET <boolean-identifier> ( <numeric-expression> ) = <boolean-expression>
          ((*numeric-vector-assigment* *string-vector-assigment* *boolean-vector-assigment*)
           (run-vector-assignment (cadr current-instruction) (caddr current-instruction) (cadddr current-instruction) hierarchy instruction-pointer env)
           (run-instruction-by-instruction-pointer (add1 instruction-pointer) hierarchy #f env))
          
          ;; LOCALS <expression-list>
          ((*locals*) (run-locals (cadr current-instruction) hierarchy instruction-pointer env)
                      (run-instruction-by-instruction-pointer (add1 instruction-pointer) hierarchy #f env))
          
          ;; DIM <identifier> ( <numeric-expression> )
          ((*dim*) (run-dim (cadr current-instruction) (caddr current-instruction) hierarchy instruction-pointer env)
                   (run-instruction-by-instruction-pointer (add1 instruction-pointer) hierarchy #f env))
          
          ;; FOR
          ((*for-to* *for-to-step*) 
           ;; If we do not come from a next...
           (if (not additional-data)
               (begin
                 ;; We first evaluate the variables...
                 (let* ((for (get-for instruction-pointer))
                        (for-ip (list-ref for 0))
                        (next-ip (list-ref for 1))
                        (var (list-ref for 2)))
                   ;; ...and update them into the global-fors structure.
                   (set-for instruction-pointer (list for-ip next-ip var (eval-expression (caddr current-instruction) hierarchy instruction-pointer env)
                                                      (eval-expression (cadddr current-instruction) hierarchy instruction-pointer env)
                                                      (if (= (vector-length (list->vector current-instruction)) 5) (eval-expression (car (cddddr current-instruction))
                                                                                                                                    hierarchy instruction-pointer env) 1 ))))
                 ;; Then, we register the index into the global environment.
                 (extend-dictionary! (cadr current-instruction) (eval-expression (caddr current-instruction) hierarchy instruction-pointer env) (car env))))
           ;; Otherwise, we do not forget to compare the start variable to the end variable according to the value of step.
           (if ((if (> (list-ref (get-for instruction-pointer) 5) 0) > <) 
                (eval-expression (list-ref (get-for instruction-pointer) 2) hierarchy instruction-pointer env) (list-ref (get-for instruction-pointer) 4))
               ;; Then we either run the first instruction of the FOR, or the first instruction at the end of the FOR depending on the start/end variables values. 
               (run-instruction-by-instruction-pointer (add1 (list-ref (get-for instruction-pointer) 1)) hierarchy #f env)
               (run-instruction-by-instruction-pointer (add1 instruction-pointer) hierarchy #f env)))
          
          ;; NEXT
          ((*next* *next-identifier*)
           ;; We update the index variable...
           (extend-dictionary! (list-ref (get-next instruction-pointer) 2)
                               (+ (eval-expression (list-ref (get-next instruction-pointer) 2) hierarchy instruction-pointer env)
                                  (list-ref (get-next instruction-pointer) 5)) (car env))
           ;; ...and run the first instruction of the FOR.
           (run-instruction-by-instruction-pointer (list-ref (get-next instruction-pointer) 0) hierarchy #t env))
          
          ;; LABEL <label>
          ((*label*) (run-instruction-by-instruction-pointer (add1 instruction-pointer) hierarchy #f env))
          
          ;; GOTO <label>
          ((*goto*) (run-instruction-by-instruction-pointer (get-label-value (cadr current-instruction)) hierarchy #f env))
          
          ;; SUB <identifier> ( <parameters> )
          ((*sub*)
           ;; If we do not come from a function call...
           (if (not additional-data)
               (begin
                 ;; We bind this function with the current environment...
                 (set-car! (cdddr (get-sub instruction-pointer)) env)
                 ;; ...and jump to the end of the function.
                 (run-instruction-by-instruction-pointer (add1 (caddr (get-sub instruction-pointer))) hierarchy #f env))
               ;; Otherwise, we continue...
               (begin
                 ;; ...checking the number of given and needed arguments...
                 (if (not (= (length (caddr current-instruction))
                             (length  additional-data)))
                     ;; ...displaying an error if needed...
                     (error "[INTERPRETER] Number of arguments do not match with needed number of arguments for function" (cadr current-instruction))
                     ;; ...checking the argument types...
                     (if (argument-types-mismatch additional-data (caddr current-instruction))
                         ;; ...displaying an error if needed...
                         (error "[INTERPRETER] Argument types mismatch" (cadr current-instruction))))
                 ;; ...and running the next instruction (whithout forgetting to extend the environment).
                 (if (cadddr (get-sub instruction-pointer))
                     (run-instruction-by-instruction-pointer (add1 instruction-pointer) hierarchy #f (extend-env (caddr current-instruction) additional-data (car (cdddr (get-sub instruction-pointer)))))
                     (error "[INTERPRETER] Undeclared function" (cadr current-instruction))))))
          
          ;; END
          ((*end*) 0)
          
          ;; ENDSUB
          ((*endsub*) 0)
          
          ;; RETURN
          ((*return*) 0)
          
          ;; RETURN <expression>
          ((*return-expression*) (eval-expression (eval-expression (cadr current-instruction) hierarchy instruction-pointer env) hierarchy instruction-pointer env))
          
          ;; GRAPHICS <numeric-expression>, <numeric-expression>
          ((*graphics*) (graphics-init (eval-expression (cadr current-instruction) hierarchy instruction-pointer env)
                                       (eval-expression (caddr current-instruction) hierarchy instruction-pointer env))
                        (run-instruction-by-instruction-pointer (add1 instruction-pointer) hierarchy #f env))
          
          ;; RESIZE <numeric-expression>, <numeric-expression>
          ((*resize*) (graphics-resize (eval-expression (cadr current-instruction) hierarchy instruction-pointer env)
                                       (eval-expression (caddr current-instruction) hierarchy instruction-pointer env))
                      (run-instruction-by-instruction-pointer (add1 instruction-pointer) hierarchy #f env))
          
          ;; MOVE <numeric-expression>, <numeric-expression>
          ((*move*) (graphics-move (eval-expression (cadr current-instruction) hierarchy instruction-pointer env)
                                   (eval-expression (caddr current-instruction) hierarchy instruction-pointer env))
                    (run-instruction-by-instruction-pointer (add1 instruction-pointer) hierarchy #f env))
          
          ;; PLOT
          ((*plot*) (graphics-plot)
                    (run-instruction-by-instruction-pointer (add1 instruction-pointer) hierarchy #f env))
          
          ;; LINETO
          ((*lineto*) (graphics-line-to (eval-expression (cadr current-instruction) hierarchy instruction-pointer env)
                                        (eval-expression (caddr current-instruction) hierarchy instruction-pointer env))
                      (run-instruction-by-instruction-pointer (add1 instruction-pointer) hierarchy #f env))
          
          ;; COLOR <numeric-expression>, <numeric-expression>, <numeric-expression>
          ((*color-by-rgb-values*) (graphics-set-color-by-rgb-values (eval-expression (cadr current-instruction) hierarchy instruction-pointer env)
                                                                     (eval-expression (caddr current-instruction) hierarchy instruction-pointer env)
                                                                     (eval-expression (cadddr current-instruction) hierarchy instruction-pointer env))
                                   (run-instruction-by-instruction-pointer (add1 instruction-pointer) hierarchy #f env))
          
          ;; COLOR <string-expression>
          ((*color-by-name*) (graphics-set-color-by-name (eval-expression (cadr current-instruction) hierarchy instruction-pointer env)
                                                         (eval-expression (caddr current-instruction) hierarchy instruction-pointer env)
                                                         (eval-expression (cadddr current-instruction) hierarchy instruction-pointer env))
                             (run-instruction-by-instruction-pointer (add1 instruction-pointer) hierarchy #f env))
          
          ;; WAIT <numeric-expression>
          ((*wait*) (graphics-wait (eval-expression (cadr current-instruction) hierarchy instruction-pointer env))
                    (run-instruction-by-instruction-pointer (add1 instruction-pointer) hierarchy #f env))
          ;; RECTANGLE <numeric-expression>, <numeric-expression>
          ((*rectangle*) (graphics-rectangle (eval-expression (cadr current-instruction) hierarchy instruction-pointer env)
                                        (eval-expression (caddr current-instruction) hierarchy instruction-pointer env))
                         (run-instruction-by-instruction-pointer (add1 instruction-pointer) hierarchy #f env))
          ;; ELLIPSE <numeric-expression>, <numeric-expression>
          ((*ellipse*) (graphics-ellipse (eval-expression (cadr current-instruction) hierarchy instruction-pointer env)
                                      (eval-expression (caddr current-instruction) hierarchy instruction-pointer env))
                       (run-instruction-by-instruction-pointer (add1 instruction-pointer) hierarchy #f env))
          ;; FILLEDRECTANGLE <numeric-expression>, <numeric-expression>
          ((*filled-rectangle*) (graphics-filled-rectangle (eval-expression (cadr current-instruction) hierarchy instruction-pointer env)
                                        (eval-expression (caddr current-instruction) hierarchy instruction-pointer env))
                         (run-instruction-by-instruction-pointer (add1 instruction-pointer) hierarchy #f env))
          ;; FILLEDELLIPSE <numeric-expression>, <numeric-expression>
          ((*filled-ellipse*) (graphics-filled-ellipse (eval-expression (cadr current-instruction) hierarchy instruction-pointer env)
                                      (eval-expression (caddr current-instruction) hierarchy instruction-pointer env))
                       (run-instruction-by-instruction-pointer (add1 instruction-pointer) hierarchy #f env))))
      
      ;; Returns #t if the variable is a string variable.
      (define (string-var? var)
        (let ((str (symbol->string var)))
          (eq? (list-ref (string->list str) (sub1 (string-length str))) #\$)))
      
      ;; Returns #t if the variable is a numeric variable.
      (define (numeric-var? var)
        (let ((str (symbol->string var)))
          (not (member (list-ref (string->list str) (sub1 (string-length str))) '(#\$ #\#)))))
      
      ;; Returns #t if the variable is a boolean variable.
      (define (boolean-var? var)
        (let ((str (symbol->string var)))
          (eq? (list-ref (string->list str) (sub1 (string-length str))) #\#)))
      
      ;; This function returns true if the arguments types from the two lists mismatch.
      (define (argument-types-mismatch val-list var-list)
        ;; If the variable list is empty, the types match.
        (cond ((null? var-list) #f)
              ;; If the first elements match, check the others...
              ((and (string? (car val-list)) (string-var? (car var-list)))
               (argument-types-mismatch (cdr val-list) (cdr var-list)))
              ((and (number? (car val-list)) (numeric-var? (car var-list)))
               (argument-types-mismatch (cdr val-list) (cdr var-list)))
              ((and (boolean? (car val-list)) (boolean-var? (car var-list)))
               (argument-types-mismatch (cdr val-list) (cdr var-list)))
              ;; Otherwise, the argument types mismatch.
              (else #t)))
      
      ;; This function returns the instruction pointer corresponding to the given label.
      (define (get-label-value label)
        ;; We first get the (label ip) list and then return the instruction pointer.
        (cadr (assq label global-labels)))
      
      ;; This function gets the for list for a given ip.
      (define (get-for ip) (assq ip global-fors))
      
      ;; This function sets the global-fors list for a given ip.
      (define (set-for ip for)
        (define (sub L)
          (cond ((null? L) (error "[INTERPRETER] FOR not found"))
                ((= (caar L) ip) (cons for (cdr L)))
                (else (cons (car L) (sub (cdr L))))))
        
        ;; We update the global-fors variable using the recursive function.
        (set! global-fors (sub global-fors)))
      
      ;; This function gets the right for list, given the ip of the NEXT instruction.
      (define (get-next ip)
        (define (sub L)
          (cond ((null? L) (error "[INTERPRETER] NEXT not found"))
                ((= (cadar L) ip) (car L))
                (else (sub (cdr L)))))
        
        ;; We just call the recursive function.
        (sub global-fors))
      
      ;; This function gets the sub list for a given ip.
      (define (get-sub ip)
        (define (sub L)
          (cond ((null? L) (error "[INTERPRETER] SUB not found"))
                ((= (cadar L) ip) (car L))
                (else (sub (cdr L)))))
        
        ;; This function gets the sub list for a given ip.
        (sub global-subs))
      
      ;; This function gets the sub list given the function name and the current function.
      (define (get-sub-by-function hierarchy function)
        (define (sub L)
          ;; If the list is empty, we did not find the sub list.
          (cond ((null? L) (error "[INTERPRETER] SUB not found"))
                ;; If it is the right one, we return the sub list.
                ((ok? (caar L)) (car L))
                ;; Otherwise, we continue...
                (else (sub (cdr L)))))
        ;; This function returns true if compared is the right list.
        (define (ok? compared)
          ;; This function splits the given string into two parts, at the last "-" character.
          (define (split str)
            ;; This function returns the index of the last character.
            (define (get-last-separator str len)
              ;; We use a do loop to move backwards in the string...
              (do ((i (- len 1) (sub1 i)))
                ;; ...until we find a "-".
                ((char=? (string-ref str i) #\-) i)))
            ;; We first get the last separator...
            (let ((last-separator (get-last-separator str (string-length str))))
              ;; ...and then return the two sub-strings.
              (values (string->symbol (substring  str 0 last-separator))
                      (string->symbol (substring str (add1 last-separator))))))
          ;; We transform 'compared' into a string in order to split it. 
          (let ((compared-str (symbol->string compared)))
            ;; Then we get the two parts of the string...
            (let-values (((left right) (split compared-str)))
              ;; ...and compare the two parts with each element of hierarchy and the function,
              ;; in order to be sure that this is the right function, and that
              ;; it is located in our function, or in one of the caller functions.
              (and (member left hierarchy) (eq? right function)))))
        
        ;; We just call the recursive function...
        (sub global-subs))
      
      ;; INPUT <string>, <numeric-identifier>
      (define (run-input-numeric string numeric-identifier env)
        (begin
          (let ((value (read-from-string (send frame gui-read string))))
            (if (number? value)
                (extend-dictionary! numeric-identifier value (car env))
                (error "[INTERPRETER] Not a number" value)))))
      
      ;; INPUT <string>, <numeric-identifier>
      (define (run-input-string string string-identifier env)
        (begin
          (let ((value (read-from-string (send frame gui-read string))))
            (if (string? value)
                (extend-dictionary! string-identifier value (car env))
                (error "[INTERPRETER] Not a string" value)))))
      
      ;; INPUT <string>, <boolean-identifier>
      (define (run-input-boolean string boolean-identifier env)
        (begin
          (let ((value (read-from-string (send frame gui-read string))))
            (if (string? value)
                (extend-dictionary! boolean-identifier value (car env))
                (error "[INTERPRETER] Not a boolean" value)))))
      
      ;; LET <identifier> = <numeric-expression>
      ;; LET <string-identifier> = <string-expression>
      ;; LET <boolean-identifier> = <boolean-expression>
      (define (run-let-variable identifier numeric-expression hierarchy ip env)
        (extend-dictionary! identifier (eval-expression numeric-expression hierarchy ip env) (car env)))
      
      ;; LOCALS <expression-list>
      (define (run-locals expression-list hierarchy ip env)
        ;; If the expression list ist not empty...
        (if (not (null? expression-list))
            ;; ...we get a default value for the first variable...
            (let ((value (cond ((string-var? (car expression-list)) "")
                               ((numeric-var? (car expression-list)) 0)
                               ((boolean-var? (car expression-list)) #t))))
              ;; ...extend the dictionary with the default value... 
              (extend-dictionary! (car expression-list) value (car env))
              ;; ...and continue with the remaining list.
              (run-locals (cdr expression-list) hierarchy ip env))))
      
      ;; DIM <identifier> ( <numeric-expression> )
      (define (run-dim identifier numeric-expression hierarchy ip env)
        (extend-dictionary! identifier (make-vector (eval-expression numeric-expression hierarchy ip env)) (car env)))
      
      ;; PRINT <arguments>
      (define (run-print arguments hierarchy ip newline env)
        (send frame gui-write (format "~a" (apply string-append (print-arguments->string-list (evlis arguments hierarchy ip env) hierarchy ip env))) 'normal)
        (if newline (send frame gui-write "\n" 'normal)))
      
      ;; LET <identifier> ( <numeric-expression> ) = <numeric-expression>
      ;; LET <string-identifier> ( <numeric-expression> ) = <string-expression>
      ;; LET <boolean-identifier> ( <numeric-expression> ) = <boolean-expression>
      (define (run-vector-assignment identifier i value hierarchy ip env)
        (vector-set! (eval-expression identifier hierarchy ip env) (- (eval-expression i hierarchy ip env) 1) (eval-expression value hierarchy ip env)))
      
      ;; This function takes a print arguments list and converts it into a list of strings.
      (define (print-arguments->string-list list hierarchy ip env)
        ;; If the list is empty then we have nothing to do.
        (if (null? list) list
            ;; We evaluate the first parameter...
            (let ((first-parameter (eval-expression (car list) hierarchy ip env)))
              ;; If the first element is a string then we just continue with the list's cdr.
              (cond ((string? first-parameter) (cons first-parameter (print-arguments->string-list (cdr list) hierarchy ip env)))
                    ;; If the first element is a number, we typecast the first element to a string and continue with the list's cdr.
                    ((number? first-parameter) (cons (number->string first-parameter) (print-arguments->string-list (cdr list) hierarchy ip env)))
                    ;; If the first element is a boolean, we typecast the first element to a string and continue with the list's cdr.
                    ((boolean? first-parameter) (cons (if first-parameter "TRUE" "FALSE") (print-arguments->string-list (cdr list) hierarchy ip env)))
                    ;; Otherwise, there is an error - this should never happen as the parser is expected to have dealt with this error.
                    (else (error "[INTERPRETER] Unexpected error"))))))
      
      ;; This function evaluates an expression in the given environment.
      (define (eval-expression expression hierarchy ip env)
        ;; If the expression is either a number, a boolean or a string, then we just return it.
        (cond ((number? expression) expression)
              ((boolean? expression) expression)
              ((string? expression) expression)
              ;; If the expression is a symbol, we get its value from the environment.
              ((symbol? expression) (lookup expression env))
              ;; Othermise...
              (else 
               ;; We first check the expression type.
               (case (car expression)
                 ((*call/ref*)
                  ;; If the expression begins with *call*, it might be a reference to an element in an array : we have no way to make the difference
                  ;; between this and a 1-parameter function call.
                  (cond ((and (binding (cadr expression) env) (vector? (eval-expression (cadr expression) hierarchy ip env)))
                         ;; If it is, we get the element from the array.
                         (vector-ref (eval-expression (cadr expression) hierarchy ip env) (sub1 (eval-expression (caaddr expression) hierarchy ip env))))
                        ;; If the variable is bound, but is not a vector...
                        ((and (binding (cadr expression) env) (not (vector? (eval-expression (cadr expression) hierarchy ip env))))
                         ;; Warn the user there was an error.
                         (error "[INTERPRETER] Not an array" (cadr expression)))
                        ;; Otherwise, this is a function call.
                        (else
                         ;; Therefore, we recursively call run-instruction-by-instruction-pointer, get the function result...
                         (let ((result (run-instruction-by-instruction-pointer (cadr (get-sub-by-function hierarchy (cadr expression)))
                                                                               (insert-function hierarchy (cadr expression))
                                                                               (evlis (caddr expression) hierarchy ip env) env)))
                           ;; ...and return it.
                           result))))
                 ;; Then we check whether the expression is one of the function which cannot be translated into a function composition,
                 ;; i.e. which needs special forms, and cannot be evaluated directly using scheme's eval.
                 ;; STR$ ( <numeric-expression>, <numeric-expression> )
                 ((*function-str-pad*)
                  ;; We first get the two parameters...
                  (let* ((number (eval-expression (cadr expression) hierarchy ip env))
                         (min-length (eval-expression (caddr expression) hierarchy ip env))
                         ;; ...we convert the number into a string...
                         (string (number->string number)))
                    ;; If the new string is longer than the minimum length...
                    (if (>= (string-length string) min-length)
                        ;; ...we just return the new string...
                        string
                        ;; ...otherwise, we pad the string.
                        (string-pad string min-length))))
                 ;; ASC ( <string-expression> )
                 ((*function-asc*)
                  ;; We first get the string expression...
                  (let* ((string-expression (eval-expression (cadr expression) hierarchy ip env)))
                    ;; ...if the string is too short, we return 0...
                    (if (< (string-length string-expression) 1) 0
                        ;; ...otherwise the string needs some padding.
                        (list->string (list (string-ref string-expression 0))))))
                 ((or) (apply or$ (evlis (cdr expression) hierarchy ip env)))
                 ((and) (apply and$ (evlis (cdr expression) hierarchy ip env)))
                 ;; Otherwise, we use scheme's eval for the function (because the lexer/parser
                 ;; translation only used scheme functions in the expressions), and recursively
                 ;; evaluate all the arguments to pass them to the evaluated function.
                 (else (apply (eval (car expression)) (evlis (cdr expression) hierarchy ip env)))))))
      
      ;; This function emulates the and special form.
      (define (and$ . L)
        (if (null? L) #t
            (and (car L) (apply and$  (cdr L)))))
      
      ;; This function emulates the or special form.
      (define (or$ . L)
        (if (null? L) #f
            (and (car L) (apply or$  (cdr L)))))
      
      ;; This function evaluates an expression list.
      (define (evlis expression-list hierarchy ip env)
        ;; If the instruction list is empty, we have nothing to do.
        (if (null? expression-list)
            '()
            ;; Otherwise, we just call eval-expression on the first argument and recursively call
            ;; the procedure on the list's cdr.
            (cons (eval-expression (car expression-list) hierarchy ip env) (evlis (cdr expression-list) hierarchy ip env))))
      
      ;; This function returns the lower-case string representation of the given upper-case symbol,
      ;; or its normal representation if the symbol is already a lower-case symbol.
      (define (uppercase-symbol->string symbol)
        ;; We typecast the string to a symbol...
        (let ((str (symbol->string symbol)))
          ;; ...then we call string-lowercase!...
          (string-lowercase! str)
          ;; ...and return the string.
          str))
      
      ;; Inserts a function into a function hierarchy.
      (define (insert-function functions-hierarchy function)
        ;; We transform each symbol into a string and back again in order to be able to append them, with separators.
        (cons (string->symbol (string-append (uppercase-symbol->string (car functions-hierarchy))
                                             "-" (uppercase-symbol->string function))) functions-hierarchy))
      
    ;; Superclass initialization.
    (super-new))))
