;; ##################################################################################
;; # ============================================================================== #
;; # ABC - lexer-parser.ss                                                          #
;; # http://abc.lozi.org                                                            #
;; # Copyright (C) Lozi Jean-Pierre, 2004 - mailto:jean-pierre@lozi.org             #
;; # ============================================================================== #
;; #                                                                                #
;; # This program is free software; you can redistribute it and/or                  #
;; # modify it under the terms of the GNU General Public License                    #
;; # as published by the Free Software Foundation; either version 2                 #
;; # of the License, or (at your option) any later version.                         #
;; #                                                                                #
;; # This program is distributed in the hope that it will be useful,                #
;; # but WITHOUT ANY WARRANTY; without even the implied warranty of                 #
;; # MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                  #
;; # GNU General Public License for more details.                                   #
;; #                                                                                #
;; # You should have received a copy of the GNU General Public License              #
;; # along with this program; if not, write to the Free Software                    #
;; # Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.    #
;; #                                                                                #
;; ##################################################################################

;; This module contains both the lexer and the parser.
(module lexer-parser mzscheme
  
  ;; Inclusions
  (require (lib "lex.ss" "parser-tools")
           (lib "yacc.ss" "parser-tools")
           (lib "string.ss")
           (lib "pretty.ss"))
  
  ;; Public functions / variables
  (provide compute)
  
  ;; Debugging status
  (define debug #f)
  
  ;; =================================================================================================== #
  ;; Lexer abbreviations                                                                                 #
  ;; =================================================================================================== #
  (define-lex-abbrevs
   ;; Comments. Example : REM This is a comment!
   (comment (@ "REM" (* (^ (: #\newline #\return))) (: #\newline #\return )))
   ;; Endlines. Either #\newline or #\return.
   (endline (: #\newline #\return))
   ;; Whitespaces. Either #\tab or #\space.
   (whitespace (: #\tab #\space))
   ;; Letters. Pretty self-explanatory.
   (letter (: (- "a" "z") (- "A" "Z")))
   ;; Digits. I reckon you know what a digit is!?
   (digit (- "0" "9"))
   ;; Integers. Plenty of digits.
   (int (+ digit))
   ;; Floats. Floating-point numbers.
   (float (: (@ (+ digit) "." (* digit)) (@ "." (+ digit))))
   ;; Operators. +, -, <, > etc.
   (operator (: "=" "+" "-" "*" "/" "^" "<" ">" "<=" ">=" "<>"))
   ;; Keywords. Examples : LET, LABEL, FOR, NEXT...
   (keyword (: "LET" "LABEL" "FOR" "TO" "STEP" "NEXT" "DIM" "IF" "THEN" "ELSE" "GOTO" "LOCAL" "RETURN" "INPUT"
               "DEF" "PRINT" "END" "SUB" "LOCALS" "ENDSUB" "AND" "OR" "INT" "SQR" "ABS" "SIN" "COS"
                "FILLEDRECTANGLE" "FILLEDELLIPSE" "ASIN" "ACOS" "EXP" "LOG" "RND" "MOD" "GRAPHICS" "RESIZE" "MOVE" "PLOT" "LINETO" "COLOR" "RECTANGLE" "ELLIPSE"
               "LEFT$" "RIGHT$" "MID$" "STR$" "CHR$" "ASC" "LEN" "WAIT"))
   ;; Booleans. Either TRUE or FALSE.
   (boolean (: "TRUE" "FALSE"))
   ;; Variables containing strings. Example : string$
   (string-identifier (@ letter (* (: letter digit "_")) "$"))
   ;; Variables containing booleans. Example : boolean#
   (boolean-identifier (@ letter (* (: letter digit "_")) "#"))
   ;; Variables/Labels. A letter followed by letter/digits/underscores.
   (identifier (@ letter (* (: letter digit "_"))))
   ;; Strings. Double-quoted text.
   (str (@ "\"" (* (^ endline "\"")) "\"")))
  
  ;; Value tokens.
  (define-tokens value-tokens (num str bool string-identifier boolean-identifier identifier))
  
  ;; Empty tokens.
  (define-empty-tokens op-tokens
                       (before-endline endline eof lpar rpar & + - * / ^ < > <= >= <> = comma colon semicolon
                                       int sqr abs sin cos asin acos exp log rnd mod
                                       let label for to step next dim if then else goto return gosub
                                       input def print end sub locals endsub and or
                                       graphics resize move plot lineto color wait rectangle filledrectangle ellipse filledellipse
                                       left$ right$ mid$ str$ chr$ asc len
                                       highest-priority lowest-priority lower-than-else lower-than-less
                                       lower-than-identifier lower-than-string-identifier lower-than-boolean-identifier
                                       lower-than-semicolon))
  
  ;; A debug macro. Allows to debug the whole lexer with an easy syntax.
  (define-syntax get-lexeme-debug
    (syntax-rules ()
      ((debug type-str lexeme return-value)
       (begin (if debug (printf (string-append "[LEXER-PARSER] Read ~s as " type-str "\n") lexeme)) return-value))))
  
  ;; The lexer.
  (define get-lexeme
    (lexer
     ;; End of files.
     ((eof)           'eof)
     ("\e" 'eof)
     ;; Whitespaces. Either #\tab or #\space.
     ((+ whitespace)     (get-lexeme input-port))
     ;; Comments. Example : REM This is a comment!
     (comment            (get-lexeme input-port))
     ;; Operators. +, -, <, > etc.
     (operator           (get-lexeme-debug "an operator" lexeme (string->symbol lexeme)))
     ;; Keywords. Examples : LET, LABEL, FOR, NEXT...
     (keyword            (get-lexeme-debug "a keyword" lexeme (uppercase-string->symbol lexeme)))
     ;; Left parenthesis.
     ("("                (get-lexeme-debug "a left parenthesis" lexeme 'lpar))
     ;; Right parenthesis.
     (")"                (get-lexeme-debug "a right parenthesis" lexeme 'rpar))
     ;; Comma.
     (","                (get-lexeme-debug "a coma" lexeme 'comma))
     ;; Colon.
     (":"                (get-lexeme-debug "a colon" lexeme 'colon))
     ;; Colon.
     (";"                (get-lexeme-debug "a semicolon" lexeme 'semicolon))
     ;; Strings. Double-quoted text.
     (str                (get-lexeme-debug "a string" lexeme (token-str (substring lexeme 1 (sub1 (string-length lexeme))))))
     ;; End of lines.
     (endline            (get-lexeme-debug "an end of line" lexeme 'endline))
     ;; Booleans. Either TRUE or FALSE.
     (boolean            (get-lexeme-debug "a boolean" lexeme (token-bool (uppercase-string->symbol lexeme))))
     ;; Variables containing strings. Example : string$
     (string-identifier  (get-lexeme-debug "a string-identifier" lexeme (token-string-identifier (uppercase-string->symbol lexeme))))
     ;; Variables containing booleans. Example : boolean$
     (boolean-identifier (get-lexeme-debug "a boolean-identifier" lexeme (token-boolean-identifier (uppercase-string->symbol lexeme))))
     ;; Variables/Labels. A letter followed by letter/digits/underscores.
     (identifier         (get-lexeme-debug "a identifier" lexeme (token-identifier (uppercase-string->symbol lexeme))))
     ;; Floats/Integers.
     ((: int float)      (get-lexeme-debug "an int/float" lexeme (token-num (string->number lexeme))))
     ;; Unknown chars.
     ((- #\000 #\377)    (error "[LEXER-PARSER] Unparseable " (string->symbol lexeme)))))
  
  ;; =================================================================================================== #
  ;; Parser                                                                                              #
  ;; =================================================================================================== #
  (define parse
    (parser
     ;; The start symbol.
     (start program)
     ;; We stop parsing when we reach an eof.
     (end eof)
     ;; Used tokens.
     (tokens value-tokens op-tokens)
     ;; Error management.
     (error (lambda (a b c) (format "[LEXER-PARSER] Parse error near ~a " b)))
     
     (precs (left - +)
            (left * /)
            (right ^)
            (nonassoc lowest-priority)
            (nonassoc lower-than-less)
            (nonassoc lower-than-semicolon)
            (nonassoc < > <= >= = <> and or semicolon endline)
            (nonassoc lower-than-identifier lower-than-string-identifier lower-than-boolean-identifier)
            (nonassoc identifier string-identifier boolean-identifier)
            (nonassoc lower-than-else)
            (nonassoc else)
            (nonassoc before-endline)
            (nonassoc highest-priority))
     
     ;; The grammar.
     (grammar
      ;; A program is either...
      (program
       ;; ...and empty program...
       (() '())
       ;; ...just ends of lines...
       ((endlines) '())
       ;; ...an instruction list...
       ((instructions) $1)
       ;; ...or an instruction list preceded by ends of lines.
       ((endlines instructions) $2))
      
      ;; <instructions>
      (instructions
       ;; <instruction> endline
       ((instruction endlines) `(,$1))
       ;; <instruction> : <instruction> endline
       ((instruction colon instruction endlines) `(,$1 ,$3))
       ;; <instructions> <instruction> : <instruction> endline
       ((instructions instruction colon instruction endlines) `(,@$1 ,$2 ,$4))
       ;; <instructions> <instruction> endline
       ((instructions instruction endlines) `(,@$1 ,$2)))
      
      ;; <endlines>
      (endlines
       ;; <endline>
       ((endline) '())
       ;; <endlines> <endline>
       ((endlines endline) '()))
      
      ;; <instruction>
      (instruction
       ;; INPUT <string>, <identifier>
       ((input str comma identifier) `(*input-numeric* ,$2 ,$4))
       ;; INPUT <string>, <string-identifier>
       ((input str comma string-identifier) `(*input-string* ,$2 ,$4))
       ;; INPUT <string>, <boolean-identifier>
       ((input str comma boolean-identifier) `(*input-boolean* ,$2 ,$4))
       ;; PRINT <print-arguments>
       ((print print-arguments) (prec lower-than-semicolon) `(*print* ,@$2))
       ;; PRINT <print-arguments> ;
       ((print print-arguments semicolon) `(*print-newline* ,@$2))
       ;; IF <boolean-expression> THEN <instruction>
       ((if boolean-expression then instruction) (prec lower-than-else) `(*if-then* ,$2 ,$4))
       ;; IF <boolean-expression> THEN <instruction> ELSE <instruction>
       ((if boolean-expression then instruction else instruction) `(*if-then-else* ,$2 ,$4 ,$6))
       ;; LET <identifier> = <numeric-expression>
       ((let identifier = numeric-expression) `(*let-numeric* ,$2 ,$4))
       ;; LET <string-identifier> = <string-expression>
       ((let string-identifier = string-expression) `(*let-string* ,$2 ,$4))
       ;; LET <boolean-identifier> = <boolean-expression>
       ((let boolean-identifier = boolean-expression) `(*let-boolean* ,$2 ,$4))
       ;; LET <identifier> ( <numeric-expression> ) = <numeric-expression>
       ((let identifier lpar numeric-expression rpar = numeric-expression) `(*numeric-vector-assigment* ,$2 ,$4 ,$7))
       ;; LET <string-identifier> ( <numeric-expression> ) = <string-expression>
       ((let string-identifier lpar numeric-expression rpar = string-expression) `(*string-vector-assigment* ,$2 ,$4 ,$7))
       ;; LET <boolean-identifier> ( <numeric-expression> ) = <boolean-expression>
       ((let boolean-identifier lpar numeric-expression rpar = boolean-expression) `(*boolean-vector-assigment* ,$2 ,$4 ,$7))
       ;; DIM <identifier> ( <numeric-expression> )
       ((dim identifier lpar numeric-expression rpar) `(*dim* ,$2 ,$4))
       ;; DIM <string-identifier> ( <numeric-expression> )
       ((dim string-identifier lpar numeric-expression rpar) `(*dim* ,$2 ,$4))
       ;; DIM <boolean-identifier> ( <numeric-expression> )
       ((dim boolean-identifier lpar numeric-expression rpar) `(*dim* ,$2 ,$4))
       ;; FOR <identifier> = <numeric-expression> TO <numeric-expression>
       ((for identifier = numeric-expression to numeric-expression) `(*for-to* ,$2 ,$4 ,$6))
       ;; FOR <identifier> = <numeric-expression> TO <numeric-expression> STEP <numeric-expression>
       ((for identifier = numeric-expression to numeric-expression step numeric-expression) `(*for-to-step* ,$2 ,$4 ,$6 ,$8))
       ;; NEXT <identifier>
       ((next identifier) `(*next-identifier* ,$2))
       ;; NEXT
       ((next) `(*next*))
       ;; <identifier> ( <call-arguments> )
       ((identifier lpar call-arguments rpar) `(*eval* (*call/ref* ,$1 ,$3)))
       ;; <string-identifier> ( <call-arguments> )
       ((string-identifier lpar call-arguments rpar) `(*eval* (*call/ref* ,$1 ,$3)))
       ;; <boolean-identifier> ( <call-arguments> )
       ((boolean-identifier lpar call-arguments rpar) `(*eval* (*call/ref* ,$1 ,$3)))
       ;; LABEL <label>
       ((label identifier) `(*label* ,$2))
       ;; GOTO <label>
       ((goto identifier) `(*goto* ,$2))
       ;; SUB <identifier> ( <arguments> )
       ((sub identifier lpar arguments rpar) `(*sub* ,$2 ,$4))
       ;; SUB <string-identifier> ( <arguments> )
       ((sub string-identifier lpar arguments rpar) `(*sub* ,$2 ,$4))
       ;; SUB <boolean-identifier> ( <arguments> )
       ((sub boolean-identifier lpar arguments rpar) `(*sub* ,$2 ,$4))
       ;; LOCALS <variable-list>
       ((locals variable-list) `(*locals* ,$2))
       ;; ENDSUB
       ((endsub) `(*endsub*))
       ;; GRAPHICS <numeric-expression>, <numeric-expression>
       ((graphics numeric-expression comma numeric-expression) `(*graphics* ,$2 ,$4))
       ;; RESIZE <numeric-expression>, <numeric-expression>
       ((resize numeric-expression comma numeric-expression) `(*resize* ,$2 ,$4))
       ;; MOVE <numeric-expression>, <numeric-expression>
       ((move numeric-expression comma numeric-expression) `(*move* ,$2 ,$4))
       ;; PLOT
       ((plot) `(*plot*))
       ;; LINETO
       ((lineto numeric-expression comma numeric-expression) `(*lineto* ,$2 ,$4))
       ;; COLOR <numeric-expression>, <numeric-expression>, <numeric-expression>
       ((color numeric-expression comma numeric-expression comma numeric-expression) `(*color-by-rgb-values* ,$2 ,$4 ,$6))
       ;; COLOR <string-expression>
       ((color string-expression) `(*color-by-name* ,$2))
       ;; WAIT <numeric-expression>
       ((wait numeric-expression) `(*wait* ,$2))
       ;; RECTANGLE <numeric-expression>, <numeric-expression>
       ((rectangle numeric-expression comma numeric-expression) `(*rectangle* ,$2 ,$4))
       ;; ELLIPSE <numeric-expression>, <numeric-expression>
       ((ellipse numeric-expression comma numeric-expression) `(*ellipse* ,$2 ,$4))
       ;; FILLEDRECTANGLE <numeric-expression>, <numeric-expression>
       ((filledrectangle numeric-expression comma numeric-expression) `(*filled-rectangle* ,$2 ,$4))
       ;; FILLEDELLIPSE <numeric-expression>, <numeric-expression>
       ((filledellipse numeric-expression comma numeric-expression) `(*filled-ellipse* ,$2 ,$4))
       ;; END
       ((end) `(*end*))
       ;; RETURN
       ((return) `(*return*))
       ;; RETURN <expression>
       ((return expression) `(*return-expression* ,$2)))
      
      ;; Expressions.
      (expression
       ;; <boolean-expression>
       ((boolean-expression) $1)
       ;; <numeric-expression>
       ((numeric-expression) $1)
       ;; <string-expression>
       ((string-expression) $1))
      
      ;; Boolean expressions.
      (boolean-expression
       ;; TRUE / FALSE
       ((bool) (eq? $1 'true))
       ;; <boolean-identifier>
       ((boolean-identifier) $1)
       ;; ( <boolean-expression> )
       ((lpar boolean-expression rpar) $2)
       ;; <boolean-identifier> ( <call-arguments> )
       ((boolean-identifier lpar call-arguments rpar) `(*call/ref* ,$1 ,$3))
       ;; <numeric-expression> = <numeric-expression>
       ((numeric-expression = numeric-expression) `(= ,$1 ,$3))
       ;; <string-expression> = <string-expression>
       ((string-expression = string-expression) `(string=? ,$1 ,$3))
       ;; <numeric-expression> < <numeric-expression>
       ((numeric-expression < numeric-expression) `(< ,$1 ,$3))
       ;; <string-expression> < <string-expression>
       ((string-expression < string-expression) `(string<? ,$1 ,$3))
       ;; <numeric-expression> <= <numeric-expression>
       ((numeric-expression <= numeric-expression) `(<= ,$1 ,$3))
       ;; <string-expression> <= <string-expression>
       ((string-expression <= string-expression) `(string<=? ,$1 ,$3))
       ;; <numeric-expression> > <numeric-expression>
       ((numeric-expression > numeric-expression) `(> ,$1 ,$3))
       ;; <string-expression> > <string-expression>
       ((string-expression > string-expression) `(string>? ,$1 ,$3))
       ;; <numeric-expression> >= <numeric-expression>
       ((numeric-expression >= numeric-expression) `(>= ,$1 ,$3))
       ;; <string-expression> >= <string-expression>
       ((string-expression >= string-expression) `(string>=? ,$1 ,$3))
       ;; <numeric-expression> <> <numeric-expression>
       ((numeric-expression <> numeric-expression) `(not (= ,$1 ,$3)))
       ;; <string-expression> <> <string-expression>
       ((string-expression <> string-expression) `(not (string=? ,$1 ,$3)))
       ;; <boolean-expression> AND <boolean-expression>
       ((boolean-expression and boolean-expression) `(and ,$1 ,$3))
       ;; <boolean-expression> OR <boolean-expression>
       ((boolean-expression or boolean-expression) `(or ,$1 ,$3)))
      
      ;; ==========================================================================================
      ;; numeric-expression
      ;; ==========================================================================================
      ;; Numeric expressions.
      (numeric-expression
       ;; <number>
       ((num) $1)
       ;; <identifier>
       ((identifier) $1)
       ;; ( <numeric-expression> )
       ((lpar numeric-expression rpar) $2)
       ;; <identifier> ( <call-arguments> )
       ((identifier lpar call-arguments rpar) `(*call/ref* ,$1 ,$3))
       ;; - <numeric-expression>
       ((- numeric-expression) (prec lower-than-less) `(- ,$2))
       ;; <numeric-expression> + <numeric-expression>
       ((numeric-expression + numeric-expression) `(+ ,$1 ,$3))
       ;; <numeric-expression> - <numeric-expression>
       ((numeric-expression - numeric-expression) `(- ,$1 ,$3))
       ;; <numeric-expression> * <numeric-expression>
       ((numeric-expression * numeric-expression) `(* ,$1 ,$3))
       ;; <numeric-expression> / <numeric-expression>
       ((numeric-expression / numeric-expression) `(/ (exact->inexact ,$1) ,$3))
       ;; <numeric-expression> ^ <numeric-expression>
       ((numeric-expression ^ numeric-expression) `(expt ,$1 ,$3))
       ;; RND ( <numeric-expression> )
       ((rnd lpar numeric-expression rpar) `(/ (random (* 1000000 ,$3)) 1000000.0))
       ;; INT ( <numeric-expression> )
       ((int lpar numeric-expression rpar) `(inexact->exact (round ,$3)))
       ;; INT ( <string-expression> )
       ((int lpar string-expression rpar) `(inexact->exact (round (string->number ,$3))))
       ;; EXP ( <numeric-expression> )
       ((exp lpar numeric-expression rpar) `(exp ,$3))
       ;; LOG ( <numeric-expression> )
       ((log lpar numeric-expression rpar) `(log ,$3))
       ;; SIN ( <numeric-expression> )
       ((sin lpar numeric-expression rpar) `(sin ,$3))
       ;; COS ( <numeric-expression> )
       ((cos lpar numeric-expression rpar) `(cos ,$3))
       ;; ASIN ( <numeric-expression> )
       ((asin lpar numeric-expression rpar) `(asin ,$3))
       ;; ACOS ( <numeric-expression> )
       ((acos lpar numeric-expression rpar) `(acos ,$3))
       ;; ABS ( <numeric-expression> )
       ((abs lpar numeric-expression rpar) `(abs ,$3))
       ;; SQR ( <numeric-expression> )
       ((sqr lpar numeric-expression rpar) `(sqr ,$3))
       ;; MOD ( <numeric-expression>, <numeric-expression> )
       ((mod lpar numeric-expression comma numeric-expression rpar) `(modulo ,$3 ,$5)))
      
      ;; ===========================================================================================
      ;; string-expression
      ;; ===========================================================================================
      (string-expression
       ;; <string>
       ((str) $1)
       ;; <string-identifier>
       ((string-identifier) $1)
       ;; ( <string-expression> )
       ((lpar string-expression rpar) $2)
       ;; <string-identifier> ( <numeric-expression> )
       ((string-identifier lpar call-arguments rpar) `(*call/ref* ,$1 ,$3))
       ;; <string-expression> + <string-expression>
       ((string-expression + string-expression) `(string-append ,$1 ,$3))
       ;; LEFT$ ( <string-expression>, <numeric-expression> )
       ((left$ lpar string-expression comma numeric-expression rpar) `(substring ,$3 0 ,$5))
       ;; RIGHT$ ( <string-expression>, <numeric-expression> )
       ((right$ lpar string-expression comma numeric-expression rpar) `(substring ,$3 (- (string-length ,$3) ,$5)))
       ;; MID$ ( <string-expression>, <numeric-expression>, <numeric-expression> )
       ((mid$ lpar string-expression comma numeric-expression comma numeric-expression rpar) `(substring ,$3 (sub1 ,$5) (+ (sub1 ,$5) ,$7)))
       ;; MID$ ( <string-expression>, <numeric-expression> )
       ((mid$ lpar string-expression comma numeric-expression rpar) `(substring ,$3 (sub1 ,$5)))
       ;; STR$ ( <numeric-expression> )
       ((str$ lpar numeric-expression rpar) `(number->string ,$3))
       ;; STR$ ( <numeric-expression>, <numeric-expression> )
       ((str$ lpar numeric-expression comma numeric-expression rpar) `(*function-str-pad* ,$3 ,$5))
       ;; CHR$ ( <numeric-expression> )
       ((chr$ lpar numeric-expression rpar) `(list->string (list (integer->char (modulo ,$3 256)))))
       ;; ASC ( <string-expression> )
       ((asc lpar string-expression rpar) `(*function-asc* ,$3))
       ;; LEN ( <string-expression> )
       ((len lpar string-expression rpar) `(string-length ,$3)))
      
      ;; Arguments
      (arguments
       ;; nothing
       (() '())
       ;; <identifier>
       ((identifier) (prec lower-than-identifier) `(,$1))
       ;; <string-identifier>
       ((string-identifier) `(,$1))
       ;; <arguments>, <identifier>
       ((arguments comma identifier) `(,@$1 ,$3))
       ;; <arguments>, <string-identifier>
       ((arguments comma string-identifier) `(,@$1 ,$3)))
      
      ;; Arguments
      (call-arguments
       ;; nothing
       (() '())
       ;; <expression>
       ((expression) `(,$1))
       ;; <arguments>, <string-identifier>
       ((call-arguments comma expression) `(,@$1 ,$3)))
      
      ;; Variable lists
      (variable-list
       ;; nothing
       (() '())
       ;; <identifier>
       ((identifier) `(,$1))
       ;; <string-identifier>
       ((string-identifier) `(,$1))
       ;; <variable-list>, <identifier>
       ((variable-list comma string-identifier) `(,@$1 ,$3))
       ;; <variable-list>, <string-identifier>
       ((variable-list comma identifier) `(,@$1 ,$3)))
      
      ;; Print arguments...
      (print-arguments
       ;; ...can either be just a string expression...
       ((string-expression) `(,$1))
       ;; ...just a numeric expression...
       ((numeric-expression) `(,$1))
       ;; ...just a boolean expression...
       ((boolean-expression) `(,$1))
       ;; ...a list of both string expressions and numeric expressions ending with a string...
       ((print-arguments comma string-expression) `(,@$1 ,$3))
       ;; ...a list of both string expressions and numeric expressions ending with a numeric expression...
       ((print-arguments comma numeric-expression) `(,@$1 ,$3))
       ;; or a list of both string expressions and numeric expressions ending with a boolean expression.
       ((print-arguments comma boolean-expression) `(,@$1 ,$3))))))
  
  ;; This function returns the lowercase symbol representation of the given uppercase string.
  ;; or its normal representation if the symbol is already a lower-case symbol.
  (define (uppercase-string->symbol str)
    ;; First, we transform the uppercase string into a lowercase string...
    (string-lowercase! str)
    ;; ...then convert it into a symbol.
    (string->symbol str))
  
  ;; This function calls the lexer and the parser on the given program
  (define (compute expr)
    ;; We don't want the program to be case sensitive
    (string-uppercase! expr)
    ;; ...ask for some code...
    (let* ((input-string (open-input-string expr))
           (result (list->vector (parse (lambda () (get-lexeme input-string))))))
      ;; ...and return the result.
      (if (not (eq? result 'eof)) result))))